unit ImgLoadSave01;
(*
   ========================================================================
    " " GraphEngine.
       .
   : BitMap.PixelFormat = pf24bit
   ========================================================================
       TImageDesc  TLoadsaveImage:
   1)    (JPEG  BMP)  Image  .
   2)     Image   Clipboard.
   ========================================================================
   ()  ,  , , .
   () Source code  ..
   ========================================================================
*)

interface
uses
     //  
     Windows, Classes, Controls, Graphics, Dialogs, Forms,
     SysUtils, ExtDlgs, ComCtrls, ExtCtrls, JPEG, Clipbrd;

// ------------------------------------------------------------------------
//    
const FileFormatJPEG  = 1;     //    JPEG
      FileFormatBMP   = 2;     //    BMP
      ClipboardFormat = 128;   //    BitMap Clipboard
      FileFormatUnkn  = 255;   //    

// ------------------------------------------------------------------------
// ------------------------------------------------------------------------
type TLoadSaveImage = class(Tobject)
  private
     fInitDir     : string;               //   
     //    
     fLoadDlg     : TOpenPictureDialog;   //   
     fSaveDlg     : TSavePictureDialog;   //   
     //     
     fProgressBar : TProgressBar;         //   ProgressBar
     fMsgPanel    : TStatusPanel;         //   StatusPanel
     //    JPEG
     fJpegQuality     : byte;             // CompressionQuality  JPEG
     fJpegProgressive : boolean;          //   JPEG
     //   
     //    
     function VerifyPixFormat (RqBitMap : TBitMap) : boolean;
     //    
     function GetIdPixFormat (RqBitMap : TBitMap) : string;
     //       BitMap
     function LoadJpegPicFromFile (RqFileName  : string;
                                   RqImage     : TImage) : boolean;
     //  BMP   Image.Pictures.BitMap
     function LoadBmpPicFromFile  (RqFileName  : string;
                                   RqImage     : TImage) : boolean;
     //  Image.Pictures.BitMap  JPEG    
     function SaveJpegToFile (RqFileName : string;
                              RqImage    : TImage) : boolean;
     //    Image.Picture.Bitmap  
     function SaveBmpToFile  (RqFileName    : string;
                              RqImage       : TImage) : boolean;
  public
     //    
     constructor Create (RqProgressBar : TProgressBar;  //   ProgressBar
                         RqMsgPanel    : TStatusPanel); //   StatusPanel
     //  
     procedure Free;
     // -------------------------------------------------
     //  Image - File
     //      Image
     function LoadImgFromFile (RqFileName : string;
                               RqImg      : TImage) : boolean;
     //       Image
     function DlgLoadImgFromFile (RqImg : TImage) : boolean;
     // -------------------------------------------------
     //    Image  
     function SaveImgToFile (RqFileName : string;
                             RqImg      : TImage) : boolean;
     //     Image  
     function DlgSaveImgToFile (RqImg : TImage) : boolean;
     // -------------------------------------------------
     //  Image - Clipboard
     //   Image  Clipboard
     function CopyImageToClipboard(RqImg : TImage) : boolean;
     //   Clipboard  Image
     function PasteClipboardToImage(RqImage : TImage) : boolean;
     // -------------------------------------------------
     // 
     //    JPEG
     property JpegQuality : byte read fJpegQuality
                                 write fJpegQuality;
     property JpegProgressive : boolean read  fJpegProgressive
                                        write fJpegProgressive;

end;

// ========================================================================
// ========================================================================
implementation
// ========================================================================
// ========================================================================

//      
// FileFormatJPEG  FileFormatBMP   DlgSaveImgToFile
// ( !)
const DialogFilter = 'JPEG (*.jpg;*.jpeg)|*.jpg;*.jpeg | BMP (*.bmp)|*.bmp';

// ========================================================================
//     
// ========================================================================
//  
constructor TLoadsaveImage.Create
                  (RqProgressBar : TProgressBar;   //   ProgressBar
                   RqMsgPanel    : TStatusPanel);  //   StatusPanel;
begin
  inherited Create;
  //    OpenPictureDialog
  fLoadDlg := TOpenPictureDialog.Create(nil);
  with fLoadDlg do
  begin
    Filter := DialogFilter;
    FilterIndex := FileFormatJPEG;
    Ctl3D := True;
  end;
  fInitDir := '';                //   
  //    SavePictureDialog
  fSaveDlg := TSavePictureDialog.Create(nil);
  with fSaveDlg do
  begin
    Filter := DialogFilter;
    FilterIndex := FileFormatJPEG;
    Ctl3D := True;
  end;
  //     JPEG
  fJpegQuality := 100;
  fJpegProgressive := False;
end;
// ------------------------------------------------------------------------
//  
procedure TLoadsaveImage.Free;
begin
   fLoadDlg.FreeOnRelease;
   fSaveDlg.FreeOnRelease;
   inherited Free;
end;

// ========================================================================
//  PUBLIC   
// ========================================================================
// ------------------------------------------------------------------------
//      Image
function TLoadsaveImage.LoadImgFromFile (RqFileName : string;
                                         RqImg      : TImage) : boolean;
var wFileName : string;
    wExt      : string;
begin
  Result := False;
  wFileName := Trim(RqFileName);
  if wFileName = '' then Exit;
  if not FileExists(wFileName)
  then begin
      MessageDlg('     : '
      + #13#10 + wFileName,
      mtError, [mbOk], 0);
      Exit;
  end;
  wExt:= UpperCase(ExtractFileExt(wFileName));
  fInitDir := ExtractFileDir(wFileName);
  if (wExt = '.JPG') or (wExt = '.JPEG')
  then Result := LoadJpegPicFromFile (wFileName, RqImg)
  else if (wExt = '.BMP')
       then Result := LoadBmpPicFromFile (wFileName, RqImg)
       else MessageDlg('   : ' + wExt,
            mtError, [mbOk], 0);
end;
// ------------------------------------------------------------------------
//       Image
function TLoadsaveImage.DlgLoadImgFromFile (RqImg : TImage) : boolean;
begin
  Result := False;
  if fInitDir <> '' then fLoadDlg.InitialDir := fInitDir;
  if fLoadDlg.Execute
  then begin
    //    
    fInitDir := ExtractFileDir(fLoadDlg.FileName);
    //      Image
    Result := LoadImgFromFile (fLoadDlg.FileName, RqImg);
  end;
end;

// ------------------------------------------------------------------------
//    Image  
function TLoadsaveImage.SaveImgToFile (RqFileName : string;
                                       RqImg : TImage) : boolean;
var wFileName : string;   //   
    wExt      : string;   //   
    wURep     : word;     //   
begin
   Result := False;
   wFileName := Trim(RqFileName);
   wExt  := UpperCase(ExtractFileExt(wFileName));
   if not ((wExt = '.JPG') or (wExt = '.JPEG') or (wExt = '.BMP'))
   then begin
      MessageDlg('   : ' + wExt,
                 mtError, [mbOk], 0);
      Exit;
   end;
   if FileExists(wFileName)
   then begin
       wURep := MessageDlg('   : '
                           + #13#10 + wFileName
                           + #13#10 + ' .  ?',
                           mtInformation,[mbYes,mbNo],0);
       //   
       if wURep = mrYes
       then begin
          if (wExt = '.JPG') or (wExt = '.JPEG')
          then Result := SaveJpegToFile (wFileName, RqImg)
          else if (wExt = '.BMP')
               then Result := SaveBmpToFile (wFileName, RqImg);
       end;
   end
   else begin
       //     
       if (wExt = '.JPG') or (wExt = '.JPEG')
          then Result := SaveJpegToFile (wFileName, RqImg)
          else if (wExt = '.BMP')
               then Result := SaveBmpToFile (wFileName, RqImg);
   end;
end;

// ------------------------------------------------------------------------
//     Image  
function TLoadsaveImage.DlgSaveImgToFile (RqImg : TImage) : boolean;
var wFileName : string;    //   
    wExt : string;         //   
begin
   Result := False;
   if fSaveDlg.Execute
   then begin
      wFileName := Trim(fSaveDlg.FileName);
      wExt      := UpperCase(ExtractFileExt(wFileName));
      case fSaveDlg.FilterIndex of
        FileFormatJPEG :  begin
            //     ,    
            if not ((wExt = '.JPG') or (wExt = '.JPEG'))
            then wFileName := wFileName + '.jpg';
            Result := SaveImgToFile(wFileName, RqImg);
        end;
        FileFormatBMP  :  begin
           //     ,    
            if not (wExt = '.BMP')
            then wFileName := wFileName + '.bmp';
            Result := SaveImgToFile(wFileName, RqImg);
        end;
      end;
   end;
end;
// ========================================================================
//  PIVATE   
// ========================================================================
//    
function TLoadsaveImage.VerifyPixFormat (RqBitMap : TBitMap) : boolean;
begin
  Result := False;
  case RqBitMap.PixelFormat of
    pfDevice : Result := True;
    pf4bit   : Result := True;
    pf8bit   : Result := True;
    pf15bit  : Result := True;
    pf16bit  : Result := True;
    pf24bit  : Result := True;
    pf32bit  : Result := True;
  end;
end;
// ------------------------------------------------------------------------
//    
function TLoadsaveImage.GetIdPixFormat (RqBitMap : TBitMap) : string;
begin
  case RqBitMap.PixelFormat of
    pfDevice : Result := 'pfDevice';
    pf1bit   : Result := 'pf1bit';
    pf4bit   : Result := 'pf4bit';
    pf8bit   : Result := 'pf8bit';
    pf15bit  : Result := 'pf15bit';
    pf16bit  : Result := 'pf16bit';
    pf24bit  : Result := 'pf24bit';
    pf32bit  : Result := 'pf32bit';
    pfCustom : Result := 'pfCustom';
    else Result := 'pfCustom';
  end;
end;

// ========================================================================
//       JPEG 
// ========================================================================
//        JPEG
type
   TJPEGService = class(TObject)
     fProgress  : TProgressBar;
     //     JPEG 
     procedure Progress(Sender: TObject; Stage: TProgressStage;
               PercentDone: Byte; RedrawNow: Boolean; const R : TRect;
               const Msg: String);
end;
// ------------------------------------------------------------------------
//      JPEG 
procedure TJPEGService.Progress(Sender: TObject;
          Stage: TProgressStage; PercentDone: Byte;
          RedrawNow: Boolean; const R: TRect;
          const Msg: String);
begin
  case Stage of
  psStarting: begin
               fProgress.Position := 0;
               fProgress.Max      := 100;
             end;
  psEnding:   begin
               fProgress.Position := 0;
             end;
  psRunning:  begin
               fProgress.Position := PercentDone;
             end;
 end;
end;
// ------------------------------------------------------------------------
//  JPEG   Image.Pictures.BitMap
function  TLoadsaveImage.LoadJpegPicFromFile
                   (RqFileName    : string;
                    RqImage       : TImage) : boolean;
var WPicture  : TPicture;       //  
    JS        : TJPEGService;   //  
begin
  Result := False;
  if (RqFileName <> '') and Assigned(RqImage)
  then begin
      //   
      WPicture := TPicture.Create;
      JS       := TJPEGService.Create;
      try
         //  WPicture 
         WPicture.LoadFromFile(RqFileName);
         if WPicture.Graphic is TJpegImage then
         begin
           if fProgressBar <> nil
           then begin
               //        JPEG
                JS.fProgress := fProgressBar;
                //    
                TJpegImage(WPicture.Graphic).OnProgress := JS.Progress;
           end;
           //  JPG  BitMap
           TJpegImage(WPicture.Graphic).DIBNeeded;
           //  BitMap  ImageMain
           RqImage.Picture.Bitmap.Assign(TBitmap(WPicture.Graphic));
           //     24   
           RqImage.Picture.Bitmap.PixelFormat := pf24bit;
           //  
           if fMsgPanel <> nil
           then fMsgPanel.Text := '    ';
           Result := True;
         end;
      finally
         //   
         WPicture.Free;
         JS.Free;
      end;
  end;
end;

// ------------------------------------------------------------------------
//  Image.Pictures.BitMap  JPEG    
function TLoadsaveImage.SaveJpegToFile
                       (RqFileName    : string;
                        RqImage       : TImage
                        ) : boolean;
var JI   : TJpegImage;       //  
    JS   : TJPEGService;     //  
begin
  Result := False;
  if (RqFileName <> '') and Assigned(RqImage.Picture.Bitmap)
  then begin
     //   
     JS := TJPEGService.Create;
     JI := TJpegImage.Create;
     try
        if fProgressBar <> nil
        then begin
           JS.fProgress :=  fProgressBar;
           JI.OnProgress := JS.Progress;  //   
        end;
        JI.PixelFormat := jf24bit;        //    
        if fJpegQuality > 100
        then JI.CompressionQuality := 100
        else JI.CompressionQuality := fJpegQuality;
        JI.ProgressiveEncoding := fJpegProgressive;
        JI.Assign(RqImage.Picture.Bitmap);
        JI.SaveToFile(RqFileName);
        //  
        if fMsgPanel <> nil
        then fMsgPanel.Text := '    ';
        Result := True;
     finally
        //   
        JI.Free;
        JS.Free;
     end;
  end;
end;

// ========================================================================
//       BMP 
// ========================================================================
//  BMP   Image.Pictures.BitMap
function TLoadsaveImage.LoadBmpPicFromFile
                   (RqFileName    : string;
                    RqImage       : TImage) : boolean;
var WBitMap   : TBitMap;        //  
begin
  Result := False;
  if (RqFileName <> '') and Assigned(RqImage)
  then begin
      //   
      WBitMap   := TBitMap.Create;;
      try
         //  WBitMap 
         WBitMap.LoadFromFile(RqFileName);
         if VerifyPixFormat(WBitMap)
         then begin
            //     pf24bit
            WBitMap.PixelFormat := pf24bit;
            //     
            if WBitMap.PixelFormat = pf24bit
            then begin
             // ----------------------------------------------------------
             //  BitMap  RqImage
             RqImage.Picture.Bitmap.Assign(WBitMap);
             // ----------------------------------------------------------
             //  
             if fMsgPanel <> nil
             then fMsgPanel.Text := '    ';
             Result := True;
           end
           else
             if fMsgPanel <> nil
             then fMsgPanel.Text := '    pf24bit!';
         end
         else
           if fMsgPanel <> nil
           then fMsgPanel.Text := ' !'
                               +  '   ( '
                               +  GetIdPixFormat (WBitMap)
                               +  ' )   ';
      finally
         //   
         WBitMap.Free;
      end;
  end;
end;
// ------------------------------------------------------------------------
//    Image.Picture.Bitmap  
function TLoadsaveImage.SaveBmpToFile
                        (RqFileName    : string;
                         RqImage       : TImage) : boolean;
begin
  Result := False;
  if (RqFileName <> '') and Assigned(RqImage.Picture.Bitmap)
  then begin
    try
      RqImage.Picture.SaveToFile(RqFileName);
      //  
      if fMsgPanel <> nil
      then fMsgPanel.Text := '    ';
      Result := True;
    except
      if fMsgPanel <> nil
      then fMsgPanel.Text := '    ';
    end;
  end;
end;

// ========================================================================
//      Clipboard
// ========================================================================
//   BitMap  Clipboard
function CopyBitMapToClipboard(RqBitMap : TBitMap) : boolean;
begin
  Result := False;
  if not Assigned (RqBitMap) then Exit;
  if (RqBitMap.Width > 0) and (RqBitMap.Height > 0)
  then begin
    try
       Clipboard.Assign(RqBitMap);
       Result := True;
    except
       MessageDlg('CopyBitMapToClipboard : '
                 + '   Clipboard!',
                  mtWarning, [mbOk], 0);
    end;
  end;
end;
// ------------------------------------------------------------------------
//   Image  Clipboard
function TLoadsaveImage.CopyImageToClipboard(RqImg : TImage) : boolean;
begin
  Result := False;
  if Assigned(RqImg.Picture.Bitmap)
  then begin
    Result := CopyBitMapToClipboard (RqImg.Picture.Bitmap);
  end;
end;
// ------------------------------------------------------------------------
//   Clipboard  Image
function TLoadsaveImage.PasteClipboardToImage(RqImage : TImage) : boolean;
var WBitmap    : TBitmap;          //  Bitmap
begin
  Result := False;
  //   Clipboard  Windows BitMap
  if Clipboard.HasFormat(CF_BITMAP)
  then begin
    WBitmap := TBitmap.Create;
    try
      //  Clipboard  BitMap.
      WBitmap.Assign(Clipboard);
      if VerifyPixFormat(WBitMap)
      then begin
         //     pf24bit
         WBitmap.PixelFormat := pf24bit;
         //     
         if WBitmap.PixelFormat = pf24bit
         then begin
            //  BitMap  RqImage
            RqImage.Picture.Bitmap.Assign(WBitmap);
            if fMsgPanel <> nil
            then fMsgPanel.Text := '     Clipboard';
            Result := True;
         end
         else
            if fMsgPanel <> nil
            then fMsgPanel.Text := '    pf24bit!';
      end
      else
         if fMsgPanel <> nil
         then fMsgPanel.Text := ' !'
                             +  '   ( '
                             +  GetIdPixFormat (WBitMap)
                             +  ' )   Clipboard';
    finally
      WBitmap.Free;
    end;
  end
  else begin
    if fMsgPanel <> nil
    then fMsgPanel.Text := '  !'
                        +  '    Clipboard  BitMap';
  end;
end;

// ========================================================================
//               END OF IMPLEMENTATION
// ========================================================================
end.
